<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Group_Control_Css_Filter;
use Elementor\Controls_Manager;
use Elementor\Utils;
use Elementor\Scheme_Color;
use Elementor\Group_Control_Typography;
use Elementor\Scheme_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Box_Shadow;

class OSF_Elementor_Testimonials extends OSF_Elementor_Carousel_Base
{

    /**
     * Get widget name.
     *
     * Retrieve testimonial widget name.
     *
     * @since  1.0.0
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'opal-testimonials';
    }

    /**
     * Get widget title.
     *
     * Retrieve testimonial widget title.
     *
     * @since  1.0.0
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Opal Testimonials', 'worldlife-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve testimonial widget icon.
     *
     * @since  1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-testimonial';
    }

    public function get_categories()
    {
        return array('opal-addons');
    }

    /**
     * Register testimonial widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function _register_controls()
    {
        $this->start_controls_section(
            'section_testimonial', [
                'label' => __(
                    'Testimonials', 'worldlife-core'),
            ]
        );

        $this->add_control(
            'testimonials', [
                'label' => __(
                    'Testimonials Item', 'worldlife-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => [
                    [
                        'name' => 'testimonial_title',
                        'label' => __('Title', 'worldlife-core'),
                        'default' => 'Testimonial',
                        'type' => Controls_Manager::TEXT,
                    ],
                    [
                        'name' => 'testimonial_content',
                        'label' => __('Content', 'worldlife-core'),
                        'type' => Controls_Manager::TEXTAREA,
                        'default' => 'Click edit button to change this text. Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.',
                        'label_block' => true,
                        'rows' => '10',
                    ],
                    [
                        'name' => 'testimonial_image',
                        'label' => __('Choose Image', 'worldlife-core'),
                        'default' => ['url' => Utils::get_placeholder_image_src(),
                        ],
                        'type' => Controls_Manager::MEDIA,
                        'show_label' => false,
                    ],
                    [
                        'name' => 'testimonial_name',
                        'label' => __('Name', 'worldlife-core'),
                        'default' => 'John Doe',
                        'type' => Controls_Manager::TEXT,
                    ],
                    [
                        'name' => 'testimonial_job',
                        'label' => __('Job', 'worldlife-core'),
                        'default' => 'Designer',
                        'type' => Controls_Manager::TEXT,
                    ],
                    [
                        'name' => 'testimonial_link',
                        'label' => __('Link to', 'worldlife-core'),
                        'placeholder' => __('https://your-link.com', 'worldlife-core'),
                        'type' => Controls_Manager::URL,
                    ],

                ],
                'title_field' => '{{{ testimonial_name }}}',
            ]
        );

        $this->add_group_control(
            Elementor\Group_Control_Image_Size::get_type(), [
                'name' => 'testimonial_image',
                // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `testimonial_image_size` and `testimonial_image_custom_dimension`.
                'default' => 'full',
                'separator' => 'none',
            ]
        );

        $this->add_control(
            'alignment', [
                'label' => __(
                    'Alignment', 'worldlife-core'),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'center',
                'options' => [
                    'left' => [
                        'title' => __(
                            'Left', 'worldlife-core'),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => __(
                            'Center', 'worldlife-core'),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => __(
                            'Right', 'worldlife-core'),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'label_block' => false,
                //'prefix_class' => 'elementor-align-',
            ]
        );


        $this->add_responsive_control(
            'column', [
                'label' => __('Columns', 'worldlife-core'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 1,
                'options' => [1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5, 6 => 6],
            ]
        );

        $this->add_control(
            'testimonial_layout', [
                'label' => __(
                    'Layout', 'worldlife-core'),
                'type' => Controls_Manager::SELECT,
                'default' => 'layout_1',
                'options' => [
                    'layout_1' => __(
                        'Layout 1', 'worldlife-core'),
                    'layout_2' => __(
                        'Layout 2', 'worldlife-core'),
                ],
            ]
        );
        $this->add_control(
            'view', [
                'label' => __('View', 'worldlife-core'),
                'type' => Controls_Manager::HIDDEN,
                'default' => 'traditional',
            ]
        );
        $this->end_controls_section();


        // Wrapper

        $this->start_controls_section(
            'item_style', [
                'label' => __('Item', 'worldlife-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs('tabs_item_style');

        $this->start_controls_tab(
            'tab_itemr_normal', [
                'label' => __('Normal', 'worldlife-core'),
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(), [
                'name' => 'item_background_wrapper',
                'types' => [
                    'classic',
                    'gradient'
                ],
                'selector' => '{{WRAPPER}} .elementor-testimonial-inner, {{WRAPPER}} .layout_2 .elementor-testimonial-wap',
            ]
        );


        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(), [
                'name' => 'item_box_shadow',
                'selector' => '{{WRAPPER}} .elementor-testimonial-inner',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_wrapper_hover', [
                'label' => __('Hover', 'worldlife-core'),
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(), [
                'name' => 'item_background_wrapper_hover',
                'types' => [
                    'classic',
                    'gradient'
                ],
                'selector' => '{{WRAPPER}} .elementor-testimonial-inner:hover, {{WRAPPER}} .layout_2 .elementor-testimonial-inner:hover .elementor-testimonial-wap',
            ]
        );


        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(), [
                'name' => 'item_box_shadow_hover',
                'selector' => '{{WRAPPER}} .elementor-testimonial-inner:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'wrapper_padding', [
                'label' => __('Padding', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator' => 'before',
            ]
        );
        $this->add_responsive_control(
            'item_spacing',
            [
                'label' => __('Spacing', 'worldlife-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 200,
                    ],
                ],
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-item' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'item_border_radius', [
                'label' => __(
                    'Border Radius', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-inner' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'wrapper_transformation', [
                'label' => __('Hover Animation', 'worldlife-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'none' => 'None',
                    'move-up' => 'Move Up',
                    'move-down' => 'Move Down',
                ],
                'default' => 'none',
                'prefix_class' => 'testimonial-wrapper-transform-',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'wrapper_effect_duration', [
                'label' => __('Effect Duration', 'worldlife-core'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 500,
                ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 3000,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} elementor-testimonial-inner ' => 'transition: all {{SIZE}}ms',
                ],
            ]
        );

        $this->end_controls_section();


        // Image.
        $this->start_controls_section(
            'section_style_testimonial_image', [
                'label' => __(
                    'Image',
                    'worldlife-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'image_size', [
                'label' => __(
                    'Image Size', 'worldlife-core'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 20,
                        'max' => 200,
                    ],
                ],
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-image img' => 'width: {{SIZE}}{{UNIT}};height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );


        $this->start_controls_tabs('image_effects');

        $this->start_controls_tab(
            'normal', [
                'label' => __('Normal', 'worldlife-core'),
            ]
        );

        $this->add_control(
            'opacity', [
                'label' => __('Opacity', 'worldlife-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'max' => 1,
                        'min' => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-image img' => 'opacity: {{SIZE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Css_Filter::get_type(), [
                'name' => 'css_filters',
                'selector' => '{{WRAPPER}} .elementor-testimonial-image img',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'hover', [
                'label' => __('Hover', 'worldlife-core'),
            ]
        );

        $this->add_control(
            'opacity_hover', [
                'label' => __('Opacity', 'worldlife-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'max' => 1,
                        'min' => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-inner:hover .elementor-testimonial-image img' => 'opacity: {{SIZE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Css_Filter::get_type(), [
                'name' => 'css_filters_hover',
                'selector' => '{{WRAPPER}} .elementor-testimonial-inner:hover .elementor-testimonial-image img',
            ]
        );

        $this->add_control(
            'background_hover_transition', [
                'label' => __(
                    'Transition Duration',
                    'worldlife-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'max' => 3,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-image img' => 'transition-duration: {{SIZE}}s',
                ],
            ]
        );

        $this->add_control(
            'hover_animation', [
                'label' => __(
                    'Hover Animation', 'worldlife-core'),
                'type' => Controls_Manager::HOVER_ANIMATION,
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();


        $this->add_group_control(
            Group_Control_Border::get_type(), [
                'name' => 'image_border',
                'selector' => '{{WRAPPER}} .elementor-testimonial-image img',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'image_border_radius', [
                'label' => __(
                    'Border Radius', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_padding', [
                'label' => __(
                    'Padding', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-image img' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_margin', [
                'label' => __('Margin', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-wrapper .elementor-testimonial-image img' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Title
        $this->start_controls_section(
            'section_style_testimonial_title', [
                'label' => __(
                    'Title',
                    'worldlife-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color', [
                'label' => __(
                    'Text Color', 'worldlife-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(), [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .elementor-testimonial-title',
            ]
        );

        $this->add_responsive_control(
            'title_padding', [
                'label' => __(
                    'Padding', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_margin', [
                'label' => __('Margin', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Content
        $this->start_controls_section(
            'section_style_testimonial_style', [
                'label' => __(
                    'Content',
                    'worldlife-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'content_content_color', [
                'label' => __(
                    'Text Color', 'worldlife-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-content' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(), [
                'name' => 'content_typography',
                'selector' => '{{WRAPPER}} .elementor-testimonial-content',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(), [
                'name' => 'content_border',
                'placeholder' => '1px',
                'default' => '1px',
                'selector' => '{{WRAPPER}} .elementor-testimonial-content',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'content_radius', [
                'label' => __(
                    'Border Radius', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'content_padding', [
                'label' => __(
                    'Padding', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'content_margin', [
                'label' => __(
                    'Margin', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Name.
        $this->start_controls_section(
            'section_style_testimonial_name', [
                'label' => __(
                    'Name', 'worldlife-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'name_text_color', [
                'label' => __(
                    'Text Color', 'worldlife-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-name, {{WRAPPER}} .elementor-testimonial-name a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(), [
                'name' => 'name_typography',
                'selector' => '{{WRAPPER}} .elementor-testimonial-name',
            ]
        );

        $this->add_responsive_control(
            'name_padding', [
                'label' => __('Padding', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-name' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'name_margin', [
                'label' => __('Margin', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-name' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Job.
        $this->start_controls_section(
            'section_style_testimonial_job', [
                'label' => __(
                    'Job', 'worldlife-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'job_text_color', [
                'label' => __(
                    'Text Color', 'worldlife-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-job' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(), [
                'name' => 'job_typography',
                'selector' => '{{WRAPPER}} .elementor-testimonial-job',
            ]
        );

        $this->add_responsive_control(
            'job_padding', [
                'label' => __('Padding', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-job' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'job_margin', [
                'label' => __('Margin', 'worldlife-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-job' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Add Carousel Control
        $this->add_control_carousel();

    }

    /**
     * Render testimonial widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        if (!empty($settings['testimonials']) && is_array($settings['testimonials'])) {

            //wrapper
            $this->add_render_attribute('wrapper', 'class', 'elementor-testimonial-wrapper');
            $this->add_render_attribute('wrapper', 'class', $settings['testimonial_layout']);
            if ($settings['alignment']) {
                $this->add_render_attribute('wrapper', 'class', 'elementor-testimonial-text-align-' . $settings['alignment']);
            }
            // row
            $this->add_render_attribute('row', 'class', 'elementor-testimonial-row');

            // Item
            $this->add_render_attribute('item', 'class', 'elementor-testimonial-item');


            if ($settings['enable_carousel'] === 'yes') {
                $this->add_render_attribute('row', 'class', 'owl-carousel owl-theme');
                $carousel_settings = $this->get_carousel_settings();
                $this->add_render_attribute('row', 'data-settings', wp_json_encode($carousel_settings));
            } else {
                // Row
                $this->add_render_attribute('row', 'class', 'row');
                $this->add_render_attribute('row', 'data-elementor-columns', $settings['column']);
                if (!empty($settings['column_tablet'])) {
                    $this->add_render_attribute('row', 'data-elementor-columns-tablet', $settings['column_tablet']);
                }
                if (!empty($settings['column_mobile'])) {
                    $this->add_render_attribute('row', 'data-elementor-columns-mobile', $settings['column_mobile']
                    );
                }
                $this->add_render_attribute('item', 'class', 'column-item');
            }


            ?>
            <div <?php echo $this->get_render_attribute_string('wrapper'); ?>>
                <div <?php echo $this->get_render_attribute_string('row') ?>>
                    <?php foreach ($settings['testimonials'] as $testimonial):
                        $testimonial_name_html = $testimonial['testimonial_name'];
                        if (!empty($testimonial['testimonial_link']['url'])) {
                            $testimonial_name_html = '<a href="' . esc_url($testimonial['testimonial_link']['url']) . '">' . $testimonial_name_html . '</a>';
                        }
                        ?>
                        <div <?php echo $this->get_render_attribute_string('item'); ?>>
                            <div class="elementor-testimonial-inner">

                                <?php $this->render_image($settings, $testimonial); ?>

                                <div class="elementor-testimonial-wap">
                                    <div class="elementor-testimonial-wap-inner">
                                        <?php if (!empty($testimonial['testimonial_title'])) : ?>
                                            <div class="elementor-testimonial-title">
                                                <?php echo $testimonial['testimonial_title']; ?>
                                            </div>
                                        <?php endif; ?>

                                        <div class="elementor-testimonial-content">
                                            <?php echo $testimonial['testimonial_content']; ?>
                                        </div>

                                        <div class="elementor-testimonial-meta-inner">
                                            <div class="elementor-testimonial-details">
                                                <div class="elementor-testimonial-name"><?php echo $testimonial_name_html; ?></div>
                                                <div class="elementor-testimonial-job"><?php echo $testimonial['testimonial_job']; ?></div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php
        }
    }

    private function render_image($settings, $testimonial)
    { ?>
        <div class="elementor-testimonial-image">
            <?php
            $testimonial['testimonial_image_size'] = $settings['testimonial_image_size'];
            $testimonial['testimonial_image_custom_dimension'] = $settings['testimonial_image_custom_dimension'];
            if (!empty($testimonial['testimonial_image']['url'])) :
                $image_html = Group_Control_Image_Size::get_attachment_image_html($testimonial, 'testimonial_image');
                echo $image_html;
            endif;
            ?>
        </div>
        <?php
    }

}

$widgets_manager->register_widget_type(new OSF_Elementor_Testimonials());
